<?php

namespace Tests;

use Faker\Factory;
use App\Models\User;
use App\Models\Installation;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\SQLiteConnection;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Testing\AssertableJsonString;
use Illuminate\Contracts\Support\Responsable;
use Illuminate\Database\Schema\SQLiteBuilder;
use Database\Seeders\RolesAndPermissionsSeeder;
use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\TestCase as BaseTestCase;

abstract class TestCase extends BaseTestCase
{
    use CreatesApplication, DatabaseMigrations;

    protected function setUp() : void
    {
        $this->hotfixSqlite();
        parent::setUp();
        $this->faker = Factory::create();
        $this->seed(RolesAndPermissionsSeeder::class);
    }

    /**
     * Hot fix to remove action done
     * during dropForein calls from migrations
     * to prevent error due to the missing of
     * some SQL functions of SQLite
     */
    private function hotfixSqlite()
    {
        \Illuminate\Database\Connection::resolverFor('sqlite', function ($connection, $database, $prefix, $config) {
            return new class($connection, $database, $prefix, $config) extends SQLiteConnection {
                public function getSchemaBuilder()
                {
                    if ($this->schemaGrammar === null) {
                        $this->useDefaultSchemaGrammar();
                    }

                    return new class($this) extends SQLiteBuilder {
                        protected function createBlueprint($table, \Closure $callback = null)
                        {
                            return new class($table, $callback) extends Blueprint {
                                public function dropForeign($index)
                                {
                                    return $this; // Don't do a drop foreign cause SQLite doesn't support it
                                }

                                public function dropMorphs($name, $indexName = null)
                                {
                                    return $this->dropColumn("{$name}_type", "{$name}_id");
                                }
                            };
                        }
                    };
                }
            };
        });
    }

    /**
     * Make a test request with the installation header
     * automatically set
     *
     * @param \App\Models\Installation $installation
     * @param \App\Models\User $userAssociated
     *
     * @return self
     */
    protected function withInstallationHeader(Installation $installation = null, User $userAssociated = null)
    {
        if ($installation == null) {
            $installation = $this->createInstallation();
        }
        if ($userAssociated != null) {
            $installation->user_id = $userAssociated->id;
            $installation->save();
        }

        return $this->withHeader(config('app.installation_header_name'), $installation->uuid);
    }

    /**
     * Get the installation header
     * that must be added to the request's header
     *
     * @param Installation $installation
     *
     * @return array
     */
    protected function getInstallationHeader(Installation $installation = null)
    {
        $installation = $installation ?? $this->createInstallation();

        return [
            config('app.installation_header_name') => $installation->uuid,
        ];
    }

    /**
     * Get or create a new \App\Models\Installation
     *
     * @return void
     */
    protected function createInstallation()
    {
        $installation = Installation::whereNotNull('uuid')->first();
        if ($installation == null) {
            $installation = Installation::factory()->create();
        }

        return $installation;
    }

    /**
     * Assert a Responsable
     * Useful to test the response of each
     * withtout necessarly making a request
     *
     * @param Responsable $responsable
     * @param array $structure the expected structure of the response
     *
     * @return void
     */
    public function assertResponsable(Responsable $responsable, array $structure)
    {
        $request = new FormRequest();

        $request->headers->set('Accept', 'application/json');
        $response = $responsable->toResponse($request);
        $assert   = new AssertableJsonString(json_encode($response));
        $assert->assertStructure($structure);
    }

    /**
     * Assert a json/assoc array
     *
     * @param array $assoc the json/array that must be asserted
     * @param array $match the expected structure
     *
     * @return void
     */
    public function assertAssocArray(array $assoc, array $match)
    {
        $assert = new AssertableJsonString(json_encode($assoc));
        $assert->assertStructure($match);
    }
}
