<?php

namespace App\Services\Proxies;

use Illuminate\Support\Facades\Http;

/**
 * OAuthProxy
 * Class used to make OAuth authentication request
 *
 * Could be added to `\App\Providers\AppServiceProvider`
 * to be accessible everywhere withtout being initialized
 * each time
 */
class OAuthProxy
{
    /**
     * The Passport ClientID
     *
     * @var string
     */
    private string $passportClientId;

    /**
     * The Passport ClientSecret
     *
     * @var string
     */
    private string $passportClientSecret;

    /**
     *  __Construct method
     *
     * @param string $passportClientId
     * @param string $passportClientSecret
     */
    public function __construct(string $passportClientId, string $passportClientSecret)
    {
        $this->passportClientId = $passportClientId;
        $this->passportClientSecret = $passportClientSecret;
    }

    /**
     * Make internal authentication request through Passport.
     *
     * @param  string $email
     * @param  string $password
     *
     * @return \Illuminate\Http\Client\Response
     */
    public function redirectToPasswordGrant(string $email, string $password)
    {
        return Http::withHeaders([
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ])->post(route('passport.token'), [
            'grant_type' => 'password',
            'client_id' => $this->passportClientId,
            'client_secret' => $this->passportClientSecret,
            'username' => $email,
            'password' => $password,
            'scope' => null,
        ]);
    }

    /**
     * Make internal authentication request through Passport
     * to refresh the token.
     *
     * @param  string                  $refresh_token
     *
     * @return \Illuminate\Http\Client\Response
     */
    public function redirectToRefreshToken(string $refresh_token)
    {
        return Http::withHeaders([
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ])->post(route('passport.token'), [
            'grant_type' => 'refresh_token',
            'client_id' => $this->passportClientId,
            'client_secret' => $this->passportClientSecret,
            'refresh_token' => $refresh_token,
        ]);
    }
}
