<?php

namespace App\Models;

use Carbon\Carbon;
use A17\Twill\Models\Model;
use Spatie\Image\Manipulations;
use Spatie\MediaLibrary\HasMedia;
use App\Notifications\VerifyEmail;
use App\Notifications\ResetPassword;
use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\MustVerifyEmail;
use A17\Twill\Models\Behaviors\HasFiles;
use Illuminate\Notifications\Notifiable;
use A17\Twill\Models\Behaviors\HasMedias;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Foundation\Auth\Access\Authorizable;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\MustVerifyEmail as MustVerifyEmailContract;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;

class User extends Model implements HasMedia, AuthenticatableContract, AuthorizableContract, CanResetPasswordContract, MustVerifyEmailContract
{
    use Notifiable;
    use MustVerifyEmail;
    use InteractsWithMedia;
    use Authenticatable;
    use Authorizable;
    use CanResetPassword;
    use SoftDeletes;
    use HasMedias;
    use HasFiles;

    /** @var int */
    public const AVATAR_LARGE_WIDTH = 300;

    /** @var int */
    public const AVATAR_LARGE_HEIGHT = 300;

    /** @var int */
    public const AVATAR_THUMBNAIL_WIDTH = 128;

    /** @var int */
    public const AVATAR_THUMBNAIL_HEIGHT = 128;

    /** @var array */
    public const STEP_LIST = [
        'chapter1',
        'chapter2',
        'chapter3',
        'chapter4',
        'chapter5',
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'first_name', 'last_name', 'author_name', 'description', 'email', 'birth_date', 'password', 'step',
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
    ];

    /** @var array */
    public array $mediasParams = [
        'cover' => [
            'desktop' => [
                [
                    'name' => 'desktop',
                    'ratio' => 16 / 9,
                ],
            ],
            'mobile' => [
                [
                    'name' => 'mobile',
                    'ratio' => 1,
                ],
            ],
            'flexible' => [
                [
                    'name' => 'free',
                    'ratio' => 0,
                ],
                [
                    'name' => 'landscape',
                    'ratio' => 16 / 9,
                ],
                [
                    'name' => 'portrait',
                    'ratio' => 3 / 5,
                ],
            ],
        ],
    ];

    public function toDataObjectArray() : array
    {
        return [
            'id' => $this->id,
            'firstName' => $this->first_name,
            'lastName' => $this->last_name,
            'email' => $this->email,
            'authorName' => $this->author_name ? $this->author_name : '',
            'description' => $this->description,
            'birthDate' => $this->birth_date,
            'step' => $this->step,
            'userAbstract' => $this->userAbstract ? $this->userAbstract->toDataObjectArray() : null,
            'userManuscript' => $this->userManuscript ? $this->userManuscript->toDataObjectArray() : null,
            'userParentsContact' => $this->userParentsContact ? $this->userParentsContact->toDataObjectArray() : null,
        ];
    }

    public function registerMediaCollections() : void
    {
        $this->addMediaCollection('users-avatars')->singleFile();
    }

    public function registerMediaConversions(Media $media = null) : void
    {
        $this->addMediaConversion('large')
            ->performOnCollections('users-avatars')
            ->width(static::AVATAR_LARGE_WIDTH)
            ->fit(Manipulations::FIT_CROP, static::AVATAR_LARGE_WIDTH, static::AVATAR_LARGE_HEIGHT);

        $this->addMediaConversion('thumbnail')
            ->performOnCollections('users-avatars')
            ->width(static::AVATAR_THUMBNAIL_WIDTH)
            ->fit(Manipulations::FIT_CROP, static::AVATAR_THUMBNAIL_WIDTH, static::AVATAR_THUMBNAIL_HEIGHT);
    }

    public function getFullNameAttribute() : string
    {
        return $this->first_name.' '.$this->last_name;
    }

    public function userAbstract() : BelongsTo
    {
        return $this->belongsTo(UserAbstract::class);
    }

    public function userManuscript() : BelongsTo
    {
        return $this->belongsTo(UserManuscript::class);
    }

    public function userParentsContact() : BelongsTo
    {
        return $this->belongsTo(UserParentsContact::class);
    }

    public function age() : int
    {
        /* @phpstan-ignore-next-line */
        return Carbon::createFromFormat('d/m/Y', $this->birth_date)->age;
    }

    /**
     * Send the email verification notification.
     *
     * @return void
     */
    public function sendEmailVerificationNotification()
    {
        $this->notify(new VerifyEmail); // my notification
    }

    /**
     * Send the password reset notification.
     *
     * @param mixed $token
     *
     * @return void
     */
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new ResetPassword($token)); // my notification
    }
}
