<?php

namespace App\Http\Controllers\Auth;

use App\Models\User;
use Illuminate\Support\Str;
use App\Http\Controllers\Controller;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\Hashing\Hasher;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Contracts\Events\Dispatcher;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Auth\CanResetPassword;
use Illuminate\Foundation\Auth\ResetsPasswords;

class ResetPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset requests
    | and uses a simple trait to include this behavior. You're free to
    | explore this trait and override any methods you wish to tweak.
    |
    */

    use ResetsPasswords;

    /**
     * @var \Illuminate\Contracts\View\Factory
     */
    protected Factory $viewFactory;

    /**
     * @var \Illuminate\Contracts\Hashing\Hasher
     */
    protected Hasher $hasher;

    /**
     * @var \Illuminate\Contracts\Events\Dispatcher
     */
    protected Dispatcher $eventDispatcher;

    /**
     * Create a new controller instance.
     *
     * @param \Illuminate\Contracts\View\Factory $viewFactory
     * @param \Illuminate\Contracts\Hashing\Hasher $hasher
     * @param \Illuminate\Contracts\Events\Dispatcher $eventDispatcher
     *
     * @return void
     */
    public function __construct(Factory $viewFactory, Hasher $hasher, Dispatcher $eventDispatcher)
    {
        $this->middleware('guest:web');
        $this->viewFactory = $viewFactory;
        $this->hasher = $hasher;
        $this->eventDispatcher = $eventDispatcher;
    }

    /**
     * Reset the given user's password.
     *
     * @param  \Illuminate\Contracts\Auth\CanResetPassword  $user
     * @param  string  $password
     *
     * @return void
     */
    protected function resetPassword(CanResetPassword $user, string $password)
    {
        $saved = false;

        if ($user instanceof User) {
            $user->password = $this->hasher->make($password);
            $user->setRememberToken(Str::random(60));

            $saved = $user->save();
        }

        if ($saved && $user instanceof Authenticatable) {
            $this->eventDispatcher->dispatch(new PasswordReset($user));
            $this->guard()->login($user);
        }
    }

    /**
     * Get the login redirect path.
     */
    protected function redirectTo() : string
    {
        return route('web.index');
    }
}
