<?php

namespace App\Http\Controllers\Auth;

use Carbon\Carbon;
use App\Models\User;
use \A17\Twill\Models\File;
use Illuminate\Support\Str;
use App\Models\UserAbstract;
use Illuminate\Http\Request;

use App\Models\UserManuscript;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\UploadedFile;
use App\Models\UserParentsContact;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Auth\Events\Registered;

use App\ViewModels\Users\UserViewModel;
use App\Http\Requests\Web\RegisterRequest;

use App\ViewModels\Auth\RegisterViewModel;
use Illuminate\Foundation\Auth\RegistersUsers;
use App\Http\Requests\Admin\UserAbstractRequest;

use App\Http\Requests\Admin\UserParentsContactRequest;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers;

    /**
     * Create a new controller instance.
     *
     *
     * @return void
     */
    public function __construct()
    {
        // $this->middleware('guest');
    }

    /**
     * Show the application registration form.
     *
     * @param Request $request
     *
     * @return \Illuminate\View\View
     */
    public function showRegistrationForm(Request $request)
    {
        if (Auth::check()) {
            return view('web.pages.register.index', new UserViewModel($request));
        }
        
        return view('web.pages.register.index', new RegisterViewModel());
    }

    /**
     * Handle a registration request for the application.
     *
     * @param  \App\Http\Requests\Web\RegisterRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(RegisterRequest $request)
    {
        $user = $this->create($request->validated());

        event(new Registered($user));

        $this->guard()->login($user);

        $user->update(['last_login_at' => Carbon::now()]);

        return (new JsonResponse)->setStatusCode(201);
    }

    /**
     * Handle a resume submit request for the application.
     *
     * @param  \App\Http\Requests\Admin\UserAbstractRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function submitAbstract(UserAbstractRequest $request)
    {
        $user = $request->user();
        $resumeData = $request->validated();
        $abstractFile = $request->file('abstract_file');

        return DB::transaction(function () use ($user, $resumeData, $abstractFile) {
            $userAbstract = UserAbstract::create([
                'story_title'=> $resumeData['story_title'],
                'literary_form'=> $resumeData['literary_form'],
                'register'=> $resumeData['register'],
                'theme'=> $resumeData['theme'],
                'text' => $resumeData['text'],
                'published' => true,
            ]);

            if ($abstractFile) {
                /** @phpstan-ignore-next-line */
                $filename = $abstractFile->getClientOriginalName();
                /** @phpstan-ignore-next-line */
                $fileSize = $abstractFile->getSize();
                $uuid = Str::uuid();

                $cleanFilename = strval(preg_replace("#\s+#i", '-', $filename));
                $fileDirectory = 'public/uploads/' . $uuid;
                
                $uuid = $uuid . '/' . $cleanFilename;
                /** @phpstan-ignore-next-line */
                $file = $this->storeFile($abstractFile, $fileDirectory, $cleanFilename, $fileSize, $uuid, 'local');

                $userAbstract->files()->attach($file->id, ['role' => 'abstract', 'locale' => config('app.locale')]);
            }

            $user->user_abstract_id = $userAbstract->id;
            $user->step = User::STEP_LIST[1];
            $user->save();

            return (new JsonResponse)->setStatusCode(201);
        });
    }

    /**
     * Handle a resume refuse request for the application.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refuseAbstract(Request $request)
    {
        $user = $request->user();
        if ($user->userAbstract) {
            $user->userAbstract->delete();
        }

        $user->user_abstract_id = null;
        $user->step = User::STEP_LIST[0];
        $user->save();

        return (new JsonResponse)->setStatusCode(201);
    }

    /**
     * Handle a resume submit request for the application.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function submitManuscript(Request $request)
    {
        $user = $request->user();
        $manuscriptFile = $request->file('manuscript_file');

        return DB::transaction(function () use ($user, $manuscriptFile) {
            $userManuscript = UserManuscript::create([
                'title' => $user->full_name . "'s Manuscript",
                'is_approved'=> null,
                'explanatory'=> '',
                'published' => true,
            ]);

            /** @phpstan-ignore-next-line */
            $filename = $manuscriptFile->getClientOriginalName();
            /* @phpstan-ignore-next-line */
            $fileSize = $manuscriptFile->getSize();

            $uuid = Str::uuid();

            $cleanFilename = strval(preg_replace("#\s+#i", '-', $filename));
            $fileDirectory = 'public/uploads/' . $uuid;
            
            $uuid = $uuid . '/' . $cleanFilename;
            /** @phpstan-ignore-next-line */
            $file = $this->storeFile($manuscriptFile, $fileDirectory, $cleanFilename, $fileSize, $uuid, 'local');

            $userManuscript->files()->attach($file->id, ['role' => 'manuscript', 'locale' => config('app.locale')]);

            $user->user_manuscript_id = $userManuscript->id;
            $user->step = User::STEP_LIST[2];
            $user->save();

            return (new JsonResponse)->setStatusCode(201);
        });
    }

    /**
     * Handle a resume refuse request for the application.
     *
     * @param  \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refuseManuscript(Request $request)
    {
        $user = $request->user();
        if ($user->userManuscript) {
            $user->userManuscript->delete();
        }

        $user->user_manuscript_id = null;
        $user->step = User::STEP_LIST[1];
        $user->save();
        
        return (new JsonResponse)->setStatusCode(201);
    }

    /**
     * Handle a resume submit request for the application.
     *
     * @param  \App\Http\Requests\Admin\UserParentsContactRequest $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function submitParentsContact(UserParentsContactRequest $request)
    {
        $user = $request->user();
        $contactData = $request->validated();

        return DB::transaction(function () use ($user, $contactData) {
            if ($user->age() < 18) {
                $userParentsContact = UserParentsContact::create([
                    'title' => $user->full_name . "'s contact",
                    'contact_person_email'=> $contactData['contact_person_email'],
                    'phone'=> $contactData['phone'],
                    'is_approved'=> null,
                    'published' => true,
                ]);
                $user->user_parents_contact_id = $userParentsContact->id;
            }

            $user->step = User::STEP_LIST[3];
            $user->save();

            return (new JsonResponse)->setStatusCode(201);
        });
    }

    /**
     * Store and create file instance after a valid request
     *
     * @param  \Illuminate\Http\UploadedFile  $requestFile
     * @param  string  $fileDirectory
     * @param  string  $filename
     * @param  float  $size
     * @param  string  $uuid
     * @param  string  $disk
     *
     * @return \A17\Twill\Models\File
     */
    public function storeFile(UploadedFile $requestFile, string $fileDirectory, string $filename, float $size, string $uuid, string $disk)
    {
        $requestFile->storeAs($fileDirectory, $filename, $disk);

        return DB::transaction(fn () => File::create([
            'uuid' => $uuid,
            'filename' => $filename,
            'size' => $size,
        ]));
    }

    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     *
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        $userData = $data;
        $userData['password'] = Hash::make($data['password']);
        $userData['step'] = User::STEP_LIST[0];

        return DB::transaction(fn () => User::create($userData));
    }
}
