<?php

namespace App\Http\Controllers\Auth;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use Illuminate\Contracts\Bus\Dispatcher;
use Illuminate\Contracts\Routing\UrlGenerator;
use Illuminate\Foundation\Auth\AuthenticatesUsers;

class LoginController extends Controller
{
    use AuthenticatesUsers {
        validateLogin as protected baseValidateLogin;
    }

    /**
     * @var \Illuminate\Contracts\Routing\UrlGenerator
     */
    protected UrlGenerator $urlGenerator;

    protected Dispatcher $dispatcher;

    /**
     * @param \Illuminate\Contracts\Routing\UrlGenerator $urlGenerator
     * @param Dispatcher $dispatcher
     */
    public function __construct(UrlGenerator $urlGenerator, Dispatcher $dispatcher)
    {
        $this->middleware('guest:web')->except('logout');
        $this->urlGenerator = $urlGenerator;
        $this->dispatcher = $dispatcher;
    }

    /**
     * Validate the user login request.
     *
     * @param  \Illuminate\Http\Request  $request
     *
     * @throws \Illuminate\Validation\ValidationException
     *
     * @return void
     */
    protected function validateLogin(Request $request)
    {
        $this->baseValidateLogin($request);

        $user = User::query()
            ->where('email', $request->email);

        if ($user->doesntExist()) {
            $this->sendFailedLoginResponse($request);
        }
    }

    /**
     * Get the login redirect path.
     */
    public function redirectTo() : string
    {
        return $this->urlGenerator->route('web.index');
    }

    /**
     * Log the user out of the application.
     *
     * @param  \Illuminate\Http\Request  $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        $this->guard()->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        if ($response = $this->loggedOut()) {
            return $response;
        }

        return $request->wantsJson()
            ? new JsonResponse([], 204)
            : redirect('/');
    }

    /**
     * The user has logged out of the application.
     *
     * @return mixed
     */
    protected function loggedOut()
    {
        return redirect()->route('web.index');
    }
}
